# Proposed Guidelines for Developers

Here is a set of basic guidelines for creating new Actions. Following them will ensure that all Actions maintain a high quality and consistent style.

## 1. Inherit from the Correct Base Class

Choose your parent class based on the action's purpose:

### Use `BaseAction`

For data-processing actions that do not interact with a live robot environment. These actions take data, process it, and
return a result.

**Good Examples:**

- ProcessInvoice
- QRCodesExtractor

### Use `BaseRobotAction`

For actions that do interact with a live robot environment (e.g., controlling the mouse/keyboard, reading the screen).
These actions require access to the `service_factory`.

**Good Examples:**

- ClickAction
- ClearAndTypeAction
- GoToUrlAction

## 2. Core Logic Belongs in `execute`

All business logic for the action must be implemented within the `execute` method. The `__init__` method should only be
used for receiving and storing the initial parameters and configuration.

---

## 3. Follow the Parameter Passing Convention

### Configuration & Identification

Parameters that define what the action is or what it will act upon should be passed to the constructor (`__init__`).

**Example:** `GoToUrlAction(url="https://aiviro.com")` or `ClickAction(element, offset_x=10, offset_y=0, ...)`.

**Execution-Time Data**: If an action needs data that is only available at runtime, pass it to the execute method. For most of our current actions, however, keeping parameter passing in `__init__` is the cleaner pattern.

## 4. Type Hint Everything

### Action Return Value

Always specify the return type using the generic. For actions that don't return anything, use `BaseRobotAction[None]`.
For actions that do, specify the type, for example `BaseRobotAction[str]` for `GetCurrentUrlAction`.

### Method Signatures

- All parameters and return values for `__init__` and `execute` must have type hints
- Use modern type hints like `|` for unions (e.g., `pathlib.Path | str`, `str | None`, ..)

---

## 5. Write Comprehensive Docstrings

Every action class must have a docstring that includes:

1. A brief, one-line summary of what the action does
2. A more detailed paragraph explaining its function
3. `:param` for each parameter in the `__init__` method
4. `:return:` to describe the data returned by the execute method
5. `:Example:` section with a clear, runnable code snippet demonstrating how to use the action

---

## 6. Keep Actions Atomic and Focused

An action should do one thing well. Avoid creating monolithic actions that perform multiple, unrelated steps.

### Good Examples

- ClickAction
- KeyPressAction
- ScrollDownAction

### Bad Examples

❌ FindAndClickSubmitButtonAndReadConfirmationText

This should be composed of three separate actions:

- FindAction
- ClickAction
- ReadTextAction
